/*:
 * @target MZ
 * @plugindesc 【遠景専用】非タイル拡大 or タイル拡大を選択可。タイル時はoriginを拡大率で補正し位置ズレ防止＋任意オフセット可。
 * @author HS
 *
 * @param DefaultScale
 * @text 既定スケール
 * @type number
 * @decimals 3
 * @min 0.10
 * @default 1.050
 *
 * @param ApplyOnMapLoad
 * @text マップ読込時に適用
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @param RenderMode
 * @text 描画方式
 * @type select
 * @option 非タイル（Sprite単枚） @value sprite
 * @option タイル（エンジン既定） @value tile
 * @default tile
 *
 * @param AnchorMode
 * @text アンカー
 * @type select
 * @option 左上固定  @value topleft
 * @option 画面中央  @value center
 * @default topleft
 *
 * @param FollowScroll
 * @text スクロールに追従（非タイル）
 * @type boolean
 * @on する
 * @off しない
 * @default true
 *
 * @param TileOriginScale
 * @text タイル時：originを拡大率で補正
 * @type boolean
 * @on する（ズレ防止）
 * @off しない
 * @default true
 * @desc タイルモードでscale>1にした際、$gameMap.parallaxOx/Oy を倍率分だけ掛けて原点を補正します。
 *
 * @param OffsetX
 * @text 追加オフセットX（px）
 * @type number
 * @decimals 0
 * @default 0
 *
 * @param OffsetY
 * @text 追加オフセットY（px）
 * @type number
 * @decimals 0
 * @default 0
 *
 * @command setScale
 * @text 遠景スケール設定
 * @arg scale
 * @text 倍率
 * @type number
 * @decimals 3
 * @min 0.10
 * @default 1.050
 *
 * @command resetScale
 * @text 遠景スケールを1.0に戻す
 *
 * @command setRenderMode
 * @text 描画方式の切替
 * @arg mode
 * @text モード
 * @type select
 * @option 非タイル（Sprite単枚） @value sprite
 * @option タイル（エンジン既定） @value tile
 * @default tile
 *
 * @command setOffset
 * @text オフセット設定（px）
 * @arg x
 * @text X
 * @type number
 * @decimals 0
 * @default 0
 * @arg y
 * @text Y
 * @type number
 * @decimals 0
 * @default 0
 */

(() => {
  "use strict";
  const PN = "HS_ParallaxScaleFixed";
  const P = PluginManager.parameters(PN);
  const DEFAULT_SCALE   = Number(P["DefaultScale"] || 1.05);
  const APPLY_ON_LOAD   = String(P["ApplyOnMapLoad"] || "true") === "true";
  const RENDER_MODE     = String(P["RenderMode"] || "tile");
  const ANCHOR_MODE     = String(P["AnchorMode"] || "topleft");
  const FOLLOW_SCROLL   = String(P["FollowScroll"] || "true") === "true";
  const TILE_ORG_SCALE  = String(P["TileOriginScale"] || "true") === "true";
  let   OFFSET_X        = Number(P["OffsetX"] || 0);
  let   OFFSET_Y        = Number(P["OffsetY"] || 0);

  // Game_System（設定保持）
  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    _Game_System_initialize.call(this);
    this._hsParallaxScale    = DEFAULT_SCALE;
    this._hsParallaxMode     = RENDER_MODE;
    this._hsParallaxEnabled  = true;
  };
  Game_System.prototype.hsParallaxScale   = function(){ return this._hsParallaxScale != null ? this._hsParallaxScale : 1.0; };
  Game_System.prototype.setHsParallaxScale= function(v){ this._hsParallaxScale = Math.max(0.1, Number(v || 1.0)); };
  Game_System.prototype.hsParallaxMode    = function(){ return this._hsParallaxMode || "tile"; };
  Game_System.prototype.setHsParallaxMode = function(m){ this._hsParallaxMode = (m === "sprite") ? "sprite" : "tile"; };
  Game_System.prototype.hsParallaxEnabled = function(){ return this._hsParallaxEnabled !== false; };
  Game_System.prototype.setHsParallaxEnabled = function(b){ this._hsParallaxEnabled = !!b; };

  // Commands
  PluginManager.registerCommand(PN, "setScale", function(args){
    $gameSystem.setHsParallaxEnabled(true);
    $gameSystem.setHsParallaxScale(Number(args.scale || 1.0));
    const scn = SceneManager._scene;
    if (scn && scn._spriteset && scn._spriteset.applyHsParallaxVisual) scn._spriteset.applyHsParallaxVisual(true);
  });

  PluginManager.registerCommand(PN, "resetScale", function(){
    $gameSystem.setHsParallaxEnabled(true);
    $gameSystem.setHsParallaxScale(1.0);
    const scn = SceneManager._scene;
    if (scn && scn._spriteset && scn._spriteset.applyHsParallaxVisual) scn._spriteset.applyHsParallaxVisual(true);
  });

  PluginManager.registerCommand(PN, "setRenderMode", function(args){
    $gameSystem.setHsParallaxMode(String(args.mode || "tile"));
    const scn = SceneManager._scene;
    if (scn && scn._spriteset && scn._spriteset.rebuildHsParallax) scn._spriteset.rebuildHsParallax();
  });

  PluginManager.registerCommand(PN, "setOffset", function(args){
    OFFSET_X = Number(args.x || 0);
    OFFSET_Y = Number(args.y || 0);
    const scn = SceneManager._scene;
    if (scn && scn._spriteset && scn._spriteset.applyHsParallaxVisual) scn._spriteset.applyHsParallaxVisual(true);
  });

  // Spriteset_Map
  const _Spriteset_Map_createParallax = Spriteset_Map.prototype.createParallax;
  Spriteset_Map.prototype.createParallax = function() {
    _Spriteset_Map_createParallax.call(this);

    // 非タイル用 Sprite
    this._hs_parallaxSprite = new Sprite();
    this._hs_parallaxSprite.visible = false;
    if (this._baseSprite && this._baseSprite.addChildAt) this._baseSprite.addChildAt(this._hs_parallaxSprite, 0);
    else this.addChild(this._hs_parallaxSprite);

    this._hs_lastAppliedScale = undefined;
    this._hs_parallaxName = undefined;

    this.rebuildHsParallax();
    if (APPLY_ON_LOAD) this.applyHsParallaxVisual(true);
  };

  Spriteset_Map.prototype.rebuildHsParallax = function() {
    const useSprite = $gameSystem.hsParallaxMode() === "sprite";
    if (useSprite) {
      this.refreshHsParallaxBitmap(() => {
        const ready = !!(this._hs_parallaxSprite && this._hs_parallaxSprite.bitmap && this._hs_parallaxSprite.bitmap.isReady());
        if (ready && this._parallax) this._parallax.visible = false;
        this._hs_parallaxSprite.visible = !!ready;
        this._hs_lastAppliedScale = undefined;
        this.applyHsParallaxVisual(true);
      });
    } else {
      this._hs_parallaxSprite.visible = false;
      if (this._parallax) this._parallax.visible = true;
      this._hs_lastAppliedScale = undefined;
      this.applyHsParallaxVisual(true);
    }
  };

  // 非タイルのビットマップ差し替え
  Spriteset_Map.prototype.refreshHsParallaxBitmap = function(onReady) {
    if (!this._hs_parallaxSprite) return;
    const name = $gameMap.parallaxName();
    if (this._hs_parallaxName === name && this._hs_parallaxSprite.bitmap) {
      if (this._hs_parallaxSprite.bitmap.isReady() && onReady) onReady();
      return;
    }
    this._hs_parallaxName = name;
    if (!name) {
      this._hs_parallaxSprite.bitmap = null;
      this._hs_parallaxSprite.visible = false;
      if (this._parallax) this._parallax.visible = true;
      if (onReady) onReady();
      return;
    }
    const bmp = ImageManager.loadParallax(name);
    this._hs_parallaxSprite.bitmap = bmp;
    bmp.addLoadListener(() => { if (onReady) onReady(); });
  };

  // 見た目適用（倍率・位置）
  Spriteset_Map.prototype.applyHsParallaxVisual = function(force) {
    if (force === undefined) force = false;
    const enabled = $gameSystem.hsParallaxEnabled();
    const s = enabled ? ($gameSystem.hsParallaxScale() || 1.0) : 1.0;
    if (!force && this._hs_lastAppliedScale === s) return;

    const mode = $gameSystem.hsParallaxMode();

    if (mode === "sprite" && this._hs_parallaxSprite && this._hs_parallaxSprite.visible) {
      // ---- 非タイル（単枚拡大）----
      const spr = this._hs_parallaxSprite;
      const bmp = spr.bitmap;
      if (!bmp || !bmp.isReady()) return;

      if (ANCHOR_MODE === "center") spr.anchor.set(0.5, 0.5);
      else spr.anchor.set(0, 0);

      spr.scale.set(s, s);

      var ox = 0, oy = 0;
      if (FOLLOW_SCROLL) {
        var loopX = $gameMap._parallaxLoopX;
        var loopY = $gameMap._parallaxLoopY;
        var tileW = $gameMap.tileWidth();
        var tileH = $gameMap.tileHeight();
        ox = loopX ? ($gameMap._parallaxX * tileW) : ($gameMap.displayX() * tileW);
        oy = loopY ? ($gameMap._parallaxY * tileH) : ($gameMap.displayY() * tileH);
      }

      // 左上 or 中央基準＋任意オフセット
      if (ANCHOR_MODE === "center") {
        spr.x = Math.floor(Graphics.width  / 2 - ox) + OFFSET_X;
        spr.y = Math.floor(Graphics.height / 2 - oy) + OFFSET_Y;
      } else {
        spr.x = -Math.floor(ox) + OFFSET_X;
        spr.y = -Math.floor(oy) + OFFSET_Y;
      }

    } else if (this._parallax && this._parallax.visible) {
      // ---- タイル（既定）----
      // 1) 先にサイズ位置（アンカー）を補正
      var px = 0, py = 0;
      if (ANCHOR_MODE === "center") {
        px = -Math.floor((Graphics.width  * (s - 1)) / 2);
        py = -Math.floor((Graphics.height * (s - 1)) / 2);
      }
      this._parallax.move(px, py, Graphics.width, Graphics.height);

      // 2) スケール適用
      this._parallax.scale.set(s, s);

      // 3) origin を拡大率で補正（ズレ防止）＋任意オフセット
      var baseOx = $gameMap.parallaxOx();
      var baseOy = $gameMap.parallaxOy();
      var orgX   = TILE_ORG_SCALE ? baseOx * s : baseOx;
      var orgY   = TILE_ORG_SCALE ? baseOy * s : baseOy;
      this._parallax.origin.x = Math.floor(orgX) + OFFSET_X;
      this._parallax.origin.y = Math.floor(orgY) + OFFSET_Y;
    }

    this._hs_lastAppliedScale = s;
  };

  // 更新（名称変更・スクロール追従）
  const _Spriteset_Map_updateParallax = Spriteset_Map.prototype.updateParallax;
  Spriteset_Map.prototype.updateParallax = function() {
    _Spriteset_Map_updateParallax.call(this);

    const s = $gameSystem.hsParallaxScale() || 1.0;
    if ($gameSystem.hsParallaxMode() === "sprite") {
      var self = this;
      this.refreshHsParallaxBitmap(function(){
        self._hs_lastAppliedScale = undefined;
        self.applyHsParallaxVisual(true);
      });
      this.applyHsParallaxVisual(false);
    } else if (this._parallax && this._parallax.visible) {
      // タイル時は毎フレ origin を拡大率基準で再設定（オフセットも反映）
      var px = 0, py = 0;
      if (ANCHOR_MODE === "center") {
        px = -Math.floor((Graphics.width  * (s - 1)) / 2);
        py = -Math.floor((Graphics.height * (s - 1)) / 2);
      }
      this._parallax.move(px, py, Graphics.width, Graphics.height);

      this._parallax.scale.set(s, s);

      var baseOx = $gameMap.parallaxOx();
      var baseOy = $gameMap.parallaxOy();
      var orgX   = TILE_ORG_SCALE ? baseOx * s : baseOx;
      var orgY   = TILE_ORG_SCALE ? baseOy * s : baseOy;
      this._parallax.origin.x = Math.floor(orgX) + OFFSET_X;
      this._parallax.origin.y = Math.floor(orgY) + OFFSET_Y;
    }
  };
})();

